<?php
/* --------------------------------------------------------------
   WithdrawalSorting.php 2020-04-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Withdrawal\SqlCriteria;

use Gambio\Core\Criteria\SqlSorting;
use Webmozart\Assert\Assert;

/**
 * Class WithdrawalSorting
 *
 * @package Gambio\Admin\Withdrawal\SqlCriteria
 */
class WithdrawalSorting implements SqlSorting
{
    /**
     * @var string
     */
    private $attribute;
    
    /**
     * @var string
     */
    private $order;
    
    /**
     * @var string[]
     */
    private static $attributeColumnMapping = [
        'id'                        => 'withdrawal_id',
        'order.id'                  => 'order_id',
        'order.creationDate'        => 'order_date',
        'order.deliveryDate'        => 'delivery_date',
        'customer.id'               => 'customer_id',
        'customer.gender'           => 'customer_gender',
        'customer.firstName'        => 'customer_firstname',
        'customer.lastName'         => 'customer_lastname',
        'customer.address.street'   => 'customer_street_address',
        'customer.address.postcode' => 'customer_postcode',
        'customer.address.city'     => 'customer_city',
        'customer.address.country'  => 'customer_country',
        'customer.email'            => 'customer_email',
        'date'                      => 'withdrawal_date',
        'content'                   => 'withdrawal_content',
        'createdByAdmin'            => 'created_by_admin',
        'createdOn'                 => 'date_created',
    ];
    
    
    /**
     * WithdrawalSorting constructor.
     *
     * @param string $attribute
     * @param string $order
     */
    public function __construct(string $attribute, string $order)
    {
        $this->attribute = $attribute;
        $this->order     = $order;
    }
    
    
    /**
     * @param string $attribute
     * @param string $order
     *
     * @return WithdrawalSorting
     */
    public static function create(string $attribute, string $order = 'ASC'): WithdrawalSorting
    {
        $attributes = array_keys(self::$attributeColumnMapping);
        Assert::oneOf($attribute,
                      $attributes,
                      'Provided attribute needs to be one of: ' . implode(', ', $attributes) . '; Got: ' . $attribute);
        Assert::oneOf($order, ['asc', 'desc'], 'Provided order needs to be one of: asc, desc; Got: ' . $order);
        
        return new self($attribute, $order);
    }
    
    
    /**
     * @return string
     */
    public function attribute(): string
    {
        return $this->attribute;
    }
    
    
    /**
     * @return string
     */
    public function order(): string
    {
        return $this->order;
    }
    
    
    /**
     * @return string
     */
    public function column(): string
    {
        return self::$attributeColumnMapping[$this->attribute];
    }
}